function fig_rate_fit = plot_rate_data(conc_fit, rate_fit, conc_data, ...
    GTP_data, err_data, time, protein_str, le_str, title_plot, ...
    ind_filt, Adj_R_sq)
% This function generates a plot with the results of CRODOCIDLE model fits
% (CROwding, COoperativity and DImerization in Luminsecence Experiments).
% The plot is stored in .pdf and .tiff format in a folder /Figures, which
% gets created if it does not already exist. Plots show the rate and if
% possible, the GTP reminaing, as function of protein concentration. Data
% points are shown as diamond markers with error bars, but the marker is a
% cross for data points excluded from the fits, shown as a dashed dark blue
% line. Colormaps from [1].
%
% Inputs:
% conc_fit     = array with concentrations, columns correspond to the order
%                of protein_str, associated with the rate estimates y_fit
% rate_fit     = vector of rates based on estimated of model parameters
% conc_data    = array with concentrations, columns correspond to the order
%                of protein_str
% GTP_data     = vector of rates
% err_data     = vector with standard errors
% time         = time vector
% protein_str  = cell array with strings of the varied proteins
% le_str       = cell array of strings for the legend. Rows are for
%                different time styrings, columns for different
%                concentration strings.
% title_plot   = string with the title for the plot
% ind_filt     = Logical vector to indicate which data points are used for
%                the fit (excluded ones are indicated as plot marker 'x')
% Adj_R_sq     = Adjusted R-squared to display in the legend
%
% Outputs:
% fig_rate_fit = Figure handle to the plotted fits.
%
% Date: 31-10-2023
%
% Bibliography
% [1] Smith, N. J., Walt, S. van der, & Firing, E. (2015). Magma, inferno,
%     plasma and viridis colormaps,
%     https://github.com/BIDS/colormap/blob/master/colormaps.py.

% General settings for font name, font size and resolution
plot_font_name      = 'Arial';
plot_font_size      = 10;
print_res           = 150;

% We loop over all time points and all different concentrations of the
% second protein if applicable, so we must find the (number of) unique time
% (t) and concentration (x) values. If there is no second protein, we set
% these artificially all to 1, such that there is only 1 unique second
% protein concentration entry, which leads to a single iteration in the
% concentration loop while plotting later on.
t_uni               = unique(time);
if numel(protein_str) > 1
    conc_2          = conc_data(:, 2);
else
    conc_2          = ones(size(conc_data));
end
conc_uni            = unique(conc_2);
% Colormap from [1]
plot_colors         = useful_colormaps('viridis', max(10, 3 + 2 * numel(t_uni)));

%% Define figure and axis
% Get the current figure and check whether this could be the figure made by
% this function (it would have 'GTPase_assay_analysis' as tag). If yes, we
% can save time by only clearing the axes and legend. Otherwise, make build
% a new figure.
fig_rate_fit            = gcf;
empty_fig               = ~strcmp(get(gcf, 'Tag'), 'GTPase_assay_analysis');
% Logical that staes whether a figure is already present

if empty_fig            % In this case the figure must be made
    % Set the properties of the figure
    set(fig_rate_fit, 'Renderer', 'painters', 'PaperPositionMode', 'auto', ...
        'Units', 'inches', 'Position', [1.5 1.5 4.5 3.1], 'Tag', 'GTPase_assay_analysis');
    % Construct the axes (Rate/left, GTP remaining/right)
    ax_rate             = axes('Parent', fig_rate_fit, 'Tag', 'Rate axes');
    ax_GTP              = axes('Parent', fig_rate_fit, 'Tag', 'GTP axes');
    % Set the properties of the axes
    set([ax_rate, ax_GTP], 'Units', 'inches', 'Position', [0.6 0.5 3.3 2.3], 'FontName', plot_font_name, ...
        'FontSize', plot_font_size, 'LabelFontSizeMultiplier', 1, 'TitleFontSizeMultiplier', 1, 'NextPlot', 'add', ...
        'LineWidth', 1, 'Box', 'off', 'color', 'none');
    set(ax_GTP, 'XAxisLocation', 'top', 'YAxisLocation', 'right', 'YDir', 'reverse', 'YScale', 'log', ...
        'Box', 'off', 'XTick', [], 'YMinorTick', 'off');
    ax_rate.YLabel.String = '% GTP hydrolysis rate / GTP [h^{-1}]';
    ax_GTP.YLabel.String = 'GTP fraction';
else % In this case, we only need to find the axes and the legend and clear/delete these
    ax_rate             = findobj(fig_rate_fit, 'Tag', 'Rate axes');
    ax_GTP              = findobj(fig_rate_fit, 'Tag', 'GTP axes');
    le_fits             = findobj(fig_rate_fit, 'Tag', 'Assay legend');
    cla(ax_rate, ax_GTP, le_fits)
    delete(le_fits)
end

% Adjust x-axis (concentration) properties (label, limits)
xlabel(ax_rate, sprintf('[%s] [\\muM]', protein_str{1}));
conc_limits                = [min(conc_fit(:, 1)) ceil(max(conc_data(:, 1) + 0.1) * 10) / 10];
set([ax_rate, ax_GTP], 'XLim', conc_limits);

%% Plot data points and fit
rate_max_plot          = 0; % We need to keep track of the maximum rate
% (+ error bar) to adjust the y-axis later to fit all points + bars.
for t = 1 : numel(t_uni)                % Loop over all time points
    ind_t               = time == t_uni(t);    % Indices data with this time
    for c2 = 1 : numel(conc_uni)            % Loop over all concentrations of
        % the second protein (one iteration if no second protein)
        ind_p           = ind_t & (conc_2 == conc_uni(c2));
        % Indices data with this time and second protein concentration
        rates           = -log(GTP_data(ind_p)) ./ time(ind_p);
        % Rate data points  following GTP = exp(-rate * time)
        rates_low       = rates + log(GTP_data(ind_p) + err_data(ind_p)) ./ time(ind_p);
        rates_upp       = -log(GTP_data(ind_p) - err_data(ind_p)) ./ time(ind_p) - rates;
        % Find the bounds on the rate data
        rate_max_plot   = max(rate_max_plot, max(rates + rates_upp));
        % Recaclulate the maximum rate thus far (for the axis limit later)
        err_bar         = errorbar(conc_data(ind_p, 1), rates, rates_low, rates_upp, ...
                            'LineStyle', 'none', 'Color', plot_colors(c2 + 2 * t, :), ...
                            'Marker', 'none', 'LineWidth', 1, 'Parent', ax_rate, 'DisplayName', le_str{t, c2});
        err_bar.Annotation.LegendInformation.IconDisplayStyle = 'off';
        % Plot data points with error bars, without marker or legend entry
        % Below we add the markers, depending on whether the point
        % contributed to the fit (diamond) or not (cross)
        ind_fit         = ind_filt & ind_p;
        % Indices of data points that contributed to the fit (diamond)
        if any(ind_fit)
            rates       = -log(GTP_data(ind_fit)) ./ time(ind_fit);
            line(conc_data(ind_fit, 1), rates, 'LineStyle', 'none', 'Color', plot_colors(c2 + 2 * t, :), ...
                'Marker', 'd', 'MarkerSize', 8, 'LineWidth', 1, 'Parent', ax_rate, ...
                'DisplayName', strcat(le_str{t, c2}, ' (fitted)'));
        end
        % Indices of data points that did no contributed to the fit (cross)
        ind_non_fit     = ~ind_filt & ind_p;
        if any(ind_non_fit)
            rates       = -log(GTP_data(ind_non_fit)) ./ time(ind_non_fit);
            line(conc_data(ind_non_fit, 1), rates, 'LineStyle', 'none', 'Color', plot_colors(c2 + 2 * t, :), ...
                'Marker', 'x', 'MarkerSize', 8, 'LineWidth', 1, 'Parent', ax_rate, ...
                'DisplayName', strcat(le_str{t, c2}, ' (not fitted)'));
        end
    end
end

% Add the fit lines for all unique values of the second protein
rate_fit_lines          = cell(numel(conc_uni), 1);
for c2 = 1 : numel(conc_uni)
    ind_plot            = ((c2 - 1) * size(conc_fit, 1) / numel(conc_uni) + 1) : c2 * size(conc_fit, 1) / numel(conc_uni);
    % Indices fot the data points of the fits of the first protein
    % concentration (all in the first column of x_fit, and different fit
    % line point are concatenated vertically
    rate_fit_lines{c2}  = line(conc_fit(ind_plot, 1), rate_fit(ind_plot), ...
                            'LineStyle', '--', 'Color', plot_colors(c2 + 2, :), 'LineWidth', 1, 'Parent', ax_rate, ...
                            'DisplayName', sprintf('Fit (Adj. R^2=%0.3f)', Adj_R_sq));
    % Adjusted R-squared is in the legend entry
    % Keep the legend entry only for the first fit line
    if c2 > 1
        rate_fit_lines{c2}.Annotation.LegendInformation.IconDisplayStyle = 'off';
    end
end

%% Adjust axis properties
% x (concentration) and y(rate) tick spacing, and rate limits (all for the rate axis)
conc_tick_spacing   = round(diff(conc_limits) / 5 * 10) / 10;
rate_limits         = [0 max(1, ceil(max(rate_max_plot) * 10) / 10)];
rate_tick_spacing   = round(diff(rate_limits) / 5 * 10) / 10;
set(ax_rate, 'XLim', conc_limits, 'XTick', 0 : conc_tick_spacing : conc_limits(2), ...
    'YLim', rate_limits, 'YTick', 0 : rate_tick_spacing : rate_limits(2));

% Update the GTP remaining axis (right y-axis). The one-to-one conversion
% from rate to GTP remaining is only possible if all data points have the
% same time, as GTP = exp(-rate * time).
if numel(t_uni) == 1 % % One time point, then GTP limits follow from
    % GTP = exp(-rate * time) and then set an appropriate tick spacing
    GTP_limits                  = fliplr(exp(-ax_rate.YLim * t_uni));
    GTP_tick_spacing            = round(diff(GTP_limits) / 10 * 10) / 10;
    set(ax_GTP, 'YLim', GTP_limits, 'YTick', 0 : GTP_tick_spacing : GTP_limits(2));
    % We can make the GTP axis visible now
    ax_GTP.Visible              = 'on';
    ax_rate.Box                 = 'off';
else % Multiple time points, so rate to GTP remaining conversion is not
     % always the same, so we make the GTP remaining axis invisible and
     % complete the box of the rate axis
    ax_rate.Box                 = 'on';
    ax_GTP.Visible              = 'off';
end

% Add the title (removing underscore to avoid GTPase construct names to get
% subscripted
title(regexprep(title_plot, '_(?=[A-Z])', ' '), 'Parent', ax_rate, 'FontName', plot_font_name, 'FontSize', plot_font_size)

% Add the legend
[le_fits, le_fits_icons]        = legend(ax_rate, 'show');
le_fits_icons(end - 1).Color    = [0 0 0];
% Turn the color of the legend entry of the fit to black to generalize its
% interpretation to accomodate all plotted fit lines
set(le_fits, 'FontName', plot_font_name, 'FontSize', plot_font_size - numel(le_str) / 5, ...
    'Units', 'inches', 'Location', 'NorthWest', 'Box', 'off', 'Tag', 'Assay legend');
le_fits.Position(1 : 2)         = [ax_rate.Position(1) sum(ax_rate.Position([2 4])) - le_fits.Position(4)];
% Move the legend

% Save as .tiff and .pdf in the directory Figures, which gets created if it
% does not exist already. The file name is the title string minus 'Run '.
if exist('./Figures', 'dir') == 0
    mkdir('./Figures/')
end
if numel(protein_str) == 1 % In the case the naming includes one protein
    print(fig_rate_fit, '-dtiffn', strcat('-r', num2str(print_res)), strcat('./Figures/', ...
        sprintf('Rate fit %s sweep Run %s', protein_str{1}, char(regexp(title_plot, '(?<=Run )\w{1,}', 'match')))))
    print(fig_rate_fit, '-dpdf', strcat('-r', num2str(print_res)), strcat('./Figures/', ...
        sprintf('Rate fit %s sweep Run %s', protein_str{1}, char(regexp(title_plot, '(?<=Run )\w{1,}', 'match')))))
else                      % In the case the naming includes two proteins
    print(fig_rate_fit, '-dtiffn', strcat('-r', num2str(print_res)), strcat('./Figures/', ...
        sprintf('Rate fit %s_%s sweep Run %s', protein_str{1}, protein_str{2}, char(regexp(title_plot, '(?<=Run )\w{1,}', 'match')))))
    print(fig_rate_fit, '-dpdf', strcat('-r', num2str(print_res)), strcat('./Figures/', ...
        sprintf('Rate fit %s_%s sweep Run %s', protein_str{1}, protein_str{2}, char(regexp(title_plot, '(?<=Run )\w{1,}', 'match')))))
end